import { NextResponse } from 'next/server';
import prisma from '@/lib/prisma';

export async function GET(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
  try {
    const { id } = await params;
    if (!id) return NextResponse.json({ ok: false, error: 'Missing ID' }, { status: 400 });

    const template = await prisma.template.findUnique({
        where: { id },
    });

    if (!template) {
        return NextResponse.json({ ok: false, error: 'Template not found' }, { status: 404 });
    }

    return NextResponse.json({ ok: true, template: template.payload });
  } catch (error: any) {
    console.error('Failed to fetch template:', error);
    return NextResponse.json({ ok: false, error: 'Failed to fetch template' }, { status: 500 });
  }
}

export async function PUT(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const body = await request.json();
        const { template } = body;

        if (!id || !template) return NextResponse.json({ ok: false, error: 'Invalid data' }, { status: 400 });

        await prisma.template.update({
            where: { id },
            data: {
                kind: template.kind,
                name: template.name,
                payload: template,
            }
        });

        return NextResponse.json({ ok: true });
    } catch (error: any) {
        console.error('Failed to update template:', error);
        return NextResponse.json({ ok: false, error: 'Failed to update template' }, { status: 500 });
    }
}

export async function DELETE(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        if (!id) return NextResponse.json({ ok: false, error: 'Missing ID' }, { status: 400 });

        await prisma.template.delete({
            where: { id },
        });

        return NextResponse.json({ ok: true });
    } catch (error: any) {
        console.error('Failed to delete template:', error);
        return NextResponse.json({ ok: false, error: 'Failed to delete template' }, { status: 500 });
    }
}
